'use strict';

/* --------------------------------------------------------------
 events.js 2019-02-26
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Main Table Events
 *
 * Handles the events of the main orders table.
 */
gx.controllers.module('events', [jse.source + '/vendor/momentjs/moment.min.js', 'loading_spinner', 'modal', 'xhr'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * On Bulk Selection Change
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Boolean} propagate Whether to affect the body elements. We do not need this on "draw.dt" event.
     */
    function _onBulkSelectionChange(event) {
        var propagate = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;

        if (propagate === false) {
            return; // Do not propagate on draw event because the body checkboxes are unchecked by default.
        }

        $this.find('tbody input:checkbox').single_checkbox('checked', $(this).prop('checked')).trigger('change');
    }

    /**
     * On Table Row Click
     *
     * When a row is clicked then the row-checkbox must be toggled.
     *
     * @param {jQuery.Event} event
     */
    function _onTableRowClick(event) {
        if (!$(event.target).is('td')) {
            return;
        }

        $(this).find('input:checkbox').prop('checked', !$(this).find('input:checkbox').prop('checked')).trigger('change');
    }

    /**
     * On Table Row Checkbox Change
     *
     * Adjust the bulk actions state whenever there are changes in the table checkboxes.
     */
    function _onTableRowCheckboxChange() {
        if ($this.find('input:checkbox:checked').length > 0) {
            $this.parents('.orders').find('.bulk-action > button').removeClass('disabled');
        } else {
            $this.parents('.orders').find('.bulk-action > button').addClass('disabled');
        }
    }

    /**
     * On Cancel Order Click
     *
     * @param {jQuery.Event} event
     */
    function _onCancelOrderClick(event) {
        event.preventDefault();

        var selectedOrders = _getSelectedOrders($(this));

        // Show the order delete modal.
        $('.cancel.modal .selected-orders').text(selectedOrders.join(', '));
        $('.cancel.modal').modal('show');
    }

    /**
     * On Send Order Click.
     *
     * Sends the email order confirmations.
     *
     * @param {jQuery.Event} event jQuery event object.
     */
    function _onBulkEmailOrderClick(event) {
        var $modal = $('.bulk-email-order.modal');
        var $mailList = $modal.find('.email-list');

        var generateMailRowMarkup = function generateMailRowMarkup(data) {
            var $row = $('<div/>', { class: 'form-group email-list-item' });
            var $idColumn = $('<div/>', { class: 'col-sm-3' });
            var $emailColumn = $('<div/>', { class: 'col-sm-9' });

            var $idLabel = $('<label/>', {
                class: 'control-label id-label force-text-color-black force-text-normal-weight',
                text: data.id
            });

            var $emailInput = $('<input/>', {
                class: 'form-control email-input',
                type: 'text',
                value: data.customerEmail
            });

            $idLabel.appendTo($idColumn);
            $emailInput.appendTo($emailColumn);

            $row.append([$idColumn, $emailColumn]);
            $row.data('order', data);

            return $row;
        };

        var selectedOrders = [];

        event.preventDefault();

        $this.find('tbody input:checkbox:checked').each(function () {
            var rowData = $(this).parents('tr').data();
            selectedOrders.push(rowData);
        });

        if (selectedOrders.length) {
            $mailList.empty();
            selectedOrders.forEach(function (order) {
                return $mailList.append(generateMailRowMarkup(order));
            });
            $modal.modal('show');
        }
    }

    /**
     * On Send Invoice Click.
     *
     * Sends the email invoice.
     *
     * @param {jQuery.Event} event Fired event.
     */
    function _onBulkEmailInvoiceClick(event) {
        var $modal = $('.bulk-email-invoice.modal');
        var $mailList = $modal.find('.email-list');

        var generateMailRowMarkup = function generateMailRowMarkup(data) {
            var $row = $('<div/>', { class: 'form-group email-list-item' });
            var $idColumn = $('<div/>', { class: 'col-sm-3' });
            var $invoiceColumn = $('<div/>', { class: 'col-sm-3' });
            var $emailColumn = $('<div/>', { class: 'col-sm-6' });

            var $latestInvoiceIdInput = $('<input/>', {
                class: 'form-control latest-invoice-id',
                type: 'hidden',
                value: data.latestInvoiceId
            });

            var $idLabel = $('<label/>', {
                class: 'control-label id-label force-text-color-black force-text-normal-weight',
                text: data.id
            });

            var $invoiceLink = $('<label/>', {
                class: 'control-label id-label force-text-color-black force-text-normal-weight',
                html: data.latestInvoiceNumber ? '<a href="request_port.php?module=OrderAdmin&action=showPdf&type=invoice' + ('&invoice_number=' + data.latestInvoiceNumber + '&order_id=' + data.id + '" target="_blank">') + (data.latestInvoiceNumber + '</a>') : '-'
            });

            var $emailInput = $('<input/>', {
                class: 'form-control email-input',
                type: 'text',
                value: data.customerEmail
            });

            $idLabel.appendTo($idColumn);
            $invoiceLink.appendTo($invoiceColumn);
            $emailInput.appendTo($emailColumn);

            $row.append([$idColumn, $invoiceColumn, $emailColumn, $latestInvoiceIdInput]);
            $row.data('order', data);

            return $row;
        };

        var selectedInvoice = [];

        event.preventDefault();

        $this.find('tbody input:checkbox:checked').each(function () {
            var rowData = $(this).parents('tr').data();
            selectedInvoice.push(rowData);
        });

        if (selectedInvoice.length) {
            $mailList.empty();
            selectedInvoice.forEach(function (order) {
                return $mailList.append(generateMailRowMarkup(order));
            });
            $modal.modal('show');
        }
    }

    /**
     * Collects the IDs of the selected orders and returns them as an array.
     *
     * @param {jQuery} $target The triggering target
     *
     * @return {Number[]} array of order IDs
     */
    function _getSelectedOrders($target) {
        var selectedOrders = [];

        if ($target.parents('.bulk-action').length > 0) {
            // Fetch the selected order IDs.
            $this.find('tbody input:checkbox:checked').each(function () {
                selectedOrders.push($(this).parents('tr').data('id'));
            });
        } else {
            var rowId = $target.parents('tr').data('id');

            if (!rowId) {
                return; // No order ID was found.
            }

            selectedOrders.push(rowId);
        }

        return selectedOrders;
    }

    /**
     * On Email Invoice Click
     *
     * Display the email-invoice modal.
     */
    function _onEmailInvoiceClick() {
        var $modal = $('.email-invoice.modal');
        var rowData = $(this).parents('tr').data();
        var url = jse.core.config.get('appUrl') + '/admin/admin.php';
        var data = {
            id: rowData.id,
            do: 'OrdersModalsAjax/GetEmailInvoiceSubject',
            pageToken: jse.core.config.get('pageToken')
        };
        var invoiceNumbersHtml = '';

        $modal.find('.customer-info').text('"' + rowData.customerName + '"');
        $modal.find('.email-address').val(rowData.customerEmail);

        $modal.data('orderId', rowData.id).modal('show');

        $.ajax({ url: url, data: data, dataType: 'json' }).done(function (response) {
            $modal.attr('data-gx-widget', 'single_checkbox');

            $modal.find('.subject').val(response.subject);
            if (response.invoiceIdExists) {
                $modal.find('.invoice-num-info').addClass('hidden');
                $modal.find('.no-invoice').removeClass('hidden');
            } else {
                $modal.find('.invoice-num-info').removeClass('hidden');
                $modal.find('.no-invoice').addClass('hidden');
            }

            if (Object.keys(response.invoiceNumbers).length <= 1) {
                $modal.find('.invoice-numbers').addClass('hidden');
            } else {
                $modal.find('.invoice-numbers').removeClass('hidden');
            }

            for (var invoiceId in response.invoiceNumbers) {
                invoiceNumbersHtml += '<p><input type="checkbox" name="invoice_ids[]" value="' + invoiceId + '" checked="checked" class="invoice-numbers-checkbox" /> ' + response.invoiceNumbers[invoiceId] + '</p>';
            }

            $modal.find('.invoice-numbers-checkboxes').html(invoiceNumbersHtml);

            $modal.find('.invoice-numbers-checkbox').on('change', _onChangeEmailInvoiceCheckbox);

            gx.widgets.init($modal);
        });
    }

    /**
     * On Email Invoice Checkbox Change
     *
     * Disable send button if all invoice number checkboxes are unchecked. Otherwise enable the send button again.
     */
    function _onChangeEmailInvoiceCheckbox() {
        var $modal = $('.email-invoice.modal');

        if ($modal.find('.invoice-numbers-checkbox').length > 0) {
            if ($modal.find('.invoice-numbers-checkbox:checked').length > 0) {
                $modal.find('.send').prop('disabled', false);
            } else {
                $modal.find('.send').prop('disabled', true);
            }
        } else {
            $modal.find('.send').prop('disabled', false);
        }
    }

    /**
     * On Email Order Click
     *
     * Display the email-order modal.
     *
     * @param {jQuery.Event} event
     */
    function _onEmailOrderClick(event) {
        var $modal = $('.email-order.modal');
        var rowData = $(this).parents('tr').data();
        var dateFormat = jse.core.config.get('languageCode') === 'de' ? 'DD.MM.YY' : 'MM/DD/YY';

        $modal.find('.customer-info').text('"' + rowData.customerName + '"');
        $modal.find('.subject').val(jse.core.lang.translate('ORDER_SUBJECT', 'gm_order_menu') + rowData.id + jse.core.lang.translate('ORDER_SUBJECT_FROM', 'gm_order_menu') + moment(rowData.purchaseDate.date).format(dateFormat));
        $modal.find('.email-address').val(rowData.customerEmail);

        $modal.data('orderId', rowData.id).modal('show');
    }

    /**
     * On Change Order Status Click
     *
     * Display the change order status modal.
     *
     * @param {jQuery.Event} event
     */
    function _onChangeOrderStatusClick(event) {
        if ($(event.target).hasClass('order-status')) {
            event.stopPropagation();
        }

        var $modal = $('.status.modal');
        var rowData = $(this).parents('tr').data();
        var selectedOrders = _getSelectedOrders($(this));

        $modal.find('#status-dropdown').val(rowData ? rowData.statusId : '');

        $modal.find('#comment').val('');
        $modal.find('#notify-customer, #send-parcel-tracking-code, #send-comment').prop('checked', false).parents('.single-checkbox').removeClass('checked');

        // Show the order change status modal.
        $modal.find('.selected-orders').text(selectedOrders.join(', '));
        $modal.modal('show');
    }

    /**
     * On Add Tracking Number Click
     *
     * @param {jQuery.Event} event
     */
    function _onAddTrackingNumberClick(event) {
        var $modal = $('.add-tracking-number.modal');
        var rowData = $(event.target).parents('tr').data();

        $modal.data('orderId', rowData.id);
        $modal.modal('show');
    }

    /**
     * Opens the gm_pdf_order.php in a new tab with invoices as type $_GET argument.
     *
     * The order ids are passed as a serialized array to the oID $_GET argument.
     */
    function _onBulkDownloadInvoiceClick() {
        var orderIds = [];
        var maxAmountInvoicesBulkPdf = data.maxAmountInvoicesBulkPdf;

        $this.find('tbody input:checkbox:checked').each(function () {
            orderIds.push($(this).parents('tr').data('id'));
        });

        if (orderIds.length > maxAmountInvoicesBulkPdf) {
            var $modal = $('.bulk-error.modal');
            $modal.modal('show');

            var $invoiceMessageContainer = $modal.find('.invoices-message');
            $invoiceMessageContainer.removeClass('hidden');
            $modal.on('hide.bs.modal', function () {
                return $invoiceMessageContainer.addClass('hidden');
            });

            return;
        }

        _createBulkPdf(orderIds, 'invoice');
    }

    /**
     * Opens the gm_pdf_order.php in a new tab with packing slip as type $_GET argument.
     *
     * The order ids are passed as a serialized array to the oID $_GET argument.
     */
    function _onBulkDownloadPackingSlipClick() {
        var orderIds = [];
        var maxAmountPackingSlipsBulkPdf = data.maxAmountPackingSlipsBulkPdf;
        var $modal = void 0;
        var $packingSlipsMessageContainer = void 0;

        $this.find('tbody input:checkbox:checked').each(function () {
            orderIds.push($(this).parents('tr').data('id'));
        });

        if (orderIds.length > maxAmountPackingSlipsBulkPdf) {
            $modal = $('.bulk-error.modal');
            $modal.modal('show');
            $packingSlipsMessageContainer = $modal.find('.packing-slips-message');

            $packingSlipsMessageContainer.removeClass('hidden');

            $modal.on('hide.bs.modal', function () {
                $packingSlipsMessageContainer.addClass('hidden');
            });

            return;
        }

        _createBulkPdf(orderIds, 'packingslip');
    }

    /**
     * Creates a bulk pdf with invoices or packing slips information.
     *
     * This method will check if all the selected orders have a document and open a concatenated PDF file. If there
     * are orders that do not have any document then a modal will be shown, prompting the user to create the missing
     * documents or continue without them.
     *
     * @param {Number[]} orderIds Provide the selected order IDs.
     * @param {String} type Provide the bulk PDF type "invoice" or "packingslip".
     */
    function _createBulkPdf(orderIds, type) {
        if (type !== 'invoice' && type !== 'packingslip') {
            throw new Error('Invalid type provided: ' + type);
        }

        var url = jse.core.config.get('appUrl') + '/admin/admin.php';
        var data = {
            do: 'OrdersOverviewAjax/GetOrdersWithoutDocuments',
            pageToken: jse.core.config.get('pageToken'),
            type: type,
            orderIds: orderIds
        };

        $.getJSON(url, data).done(function (orderIdsWithoutDocument) {
            if (orderIdsWithoutDocument.exception) {
                var title = jse.core.lang.translate('error', 'messages');
                var message = jse.core.lang.translate('GET_ORDERS_WITHOUT_DOCUMENT_ERROR', 'admin_orders');
                jse.libs.modal.showMessage(title, message);
                return;
            }

            if (!orderIdsWithoutDocument.length) {
                _openBulkPdfUrl(orderIds, type); // All the selected order have documents.
                return;
            }

            // Some orders do not have documents, display the confirmation modal.
            var $modal = $('.modal.create-missing-documents');
            $modal.find('.order-ids-list').text(orderIdsWithoutDocument.join(', '));
            $modal.data({
                orderIds: orderIds,
                orderIdsWithoutDocument: orderIdsWithoutDocument,
                type: type
            }).modal('show');
        }).fail(function (jqxhr, textStatus, errorThrown) {
            var title = jse.core.lang.translate('error', 'messages');
            var message = jse.core.lang.translate('GET_ORDERS_WITHOUT_DOCUMENT_ERROR', 'admin_orders');
            jse.libs.modal.showMessage(title, message);
        });
    }

    /**
     * Create Missing Documents Proceed Handler
     *
     * This handler will be executed whenever the user proceeds through the "create-missing-documents" modal. It will
     * be resolved even if the user does not select the checkbox "create-missing-documents".
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Number[]} orderIds The selected orders to be included in the PDF.
     * @param {String} type Whether 'invoice' or 'packingslip'.
     * @param {Object} downloadPdfWindow Provide a window handle for bypassing browser's popup blocking.
     */
    function _onCreateMissingDocumentsProceed(event, orderIds, type, downloadPdfWindow) {
        _openBulkPdfUrl(orderIds, type, downloadPdfWindow);
    }

    /**
     * On Single Checkbox Ready
     *
     * This callback will use the event.data.orderIds to set the checked checkboxes after a table re-render.
     *
     * @param {jQuery.Event} event
     */
    function _onSingleCheckboxReady(event) {
        event.data.orderIds.forEach(function (id) {
            $this.find('tr#' + id + ' input:checkbox').single_checkbox('checked', true).trigger('change');
        });

        // Bulk action button should't be disabled after a datatable reload.
        if ($('tr input:checkbox:checked').length) {
            $('.bulk-action').find('button').removeClass('disabled');
        }
    }

    /**
     * Opens the URL which provides the bulk PDF for download.
     *
     * @param {Number[]} orderIds The orders to be used for the concatenated document.
     * @param {String} type Whether 'invoice' or 'packingslip'.
     */
    function _openBulkPdfUrl(orderIds, type) {
        var parameters = {
            do: 'OrdersModalsAjax/BulkPdf' + (type === 'invoice' ? 'Invoices' : 'PackingSlips'),
            pageToken: jse.core.config.get('pageToken'),
            o: orderIds
        };

        var url = jse.core.config.get('appUrl') + '/admin/admin.php?' + $.param(parameters);

        window.open(url, '_parent');

        // Keep checkboxes checked after a datatable reload.
        $this.DataTable().ajax.reload(function () {
            $this.off('single_checkbox:ready', _onSingleCheckboxReady).on('single_checkbox:ready', { orderIds: orderIds }, _onSingleCheckboxReady);
        });
        $this.orders_overview_filter('reload');
    }

    /**
     * On Packing Slip Click
     */
    function _onShowPackingSlipClick() {
        // Message modal data.
        var title = jse.core.lang.translate('TITLE_SHOW_PACKINGSLIP', 'orders');
        var message = jse.core.lang.translate('NO_PACKINGSLIP_AVAILABLE', 'orders');

        // Request data.
        var rowData = $(this).parents('tr').data();
        var url = jse.core.config.get('appUrl') + '/admin/admin.php';

        // Request parameters.
        var data = {
            id: rowData.id,
            do: 'OrdersOverviewAjax/GetLatestPackingSlip',
            pageToken: jse.core.config.get('pageToken')
        };

        // Directly open a new tab (popup blocker workaround)
        var newTab = window.open('about:blank');

        $.ajax({ dataType: 'json', url: url, data: data }).done(function (response) {
            if (response.length) {
                // Get the file name from the response.
                var filename = response[0].file;

                // Packing slip link parameters.
                var parameters = {
                    module: 'OrderAdmin',
                    action: 'showPdf',
                    type: 'packingslip',
                    file: filename
                };

                // Open package slip.
                newTab.location = jse.core.config.get('appUrl') + '/admin/request_port.php?' + $.param(parameters);
            } else {
                // No packing slip found
                newTab.close();
                jse.libs.modal.showMessage(title, message);
            }
        });
    }

    /**
     * On Invoice Create Click
     */
    function _onCreateInvoiceClick() {
        var link = $(this).attr('href');
        var $loadingSpinner = jse.libs.loading_spinner.show($this);
        var pageToken = jse.core.config.get('pageToken');
        var orderId = $(this).parents('tr').data().id;
        var url = jse.core.config.get('appUrl') + ('/admin/admin.php?do=OrdersModalsAjax/GetInvoiceCount&pageToken=' + pageToken + '&orderId=' + orderId);

        // Directly open a new tab (popup blocker workaround)
        var newTab = window.open('about:blank');

        function createInvoice() {
            newTab.location = link;
            $this.DataTable().ajax.reload(null, false);
        }

        function addInvoice() {
            window.open(link, '_blank');
            $this.DataTable().ajax.reload(null, false);
        }

        function onRequestSuccess(response) {
            var modalTitle = jse.core.lang.translate('TITLE_CREATE_INVOICE', 'orders');
            var modalMessage = jse.core.lang.translate('TEXT_CREATE_INVOICE_CONFIRMATION', 'orders');
            var modalButtons = [{
                title: jse.core.lang.translate('yes', 'buttons'),
                callback: function callback(event) {
                    closeModal(event);
                    addInvoice();
                }
            }, {
                title: jse.core.lang.translate('no', 'buttons'),
                callback: closeModal
            }];

            function closeModal(event) {
                $(event.target).parents('.modal').modal('hide');
            }

            jse.libs.loading_spinner.hide($loadingSpinner);

            if (!response.count) {
                createInvoice();
            } else {
                newTab.close();
                jse.libs.modal.showMessage(modalTitle, modalMessage, modalButtons);
            }
        }

        function onRequestFailure() {
            jse.libs.loading_spinner.hide($loadingSpinner);
            createInvoice();
        }

        jse.libs.xhr.get({ url: url }).done(onRequestSuccess).fail(onRequestFailure);
    }

    /**
     * On Invoice Link Click
     *
     * The script that generates the PDFs is changing the status of an order to "invoice-created". Thus the
     * table data need to be redrawn and the filter options to be updated.
     */
    function _onShowInvoiceClick() {
        // Message modal data.
        var title = jse.core.lang.translate('TITLE_SHOW_INVOICE', 'orders');
        var message = jse.core.lang.translate('NO_INVOICE_AVAILABLE', 'orders');

        // Request data.
        var rowData = $(this).parents('tr').data();
        var url = jse.core.config.get('appUrl') + '/admin/admin.php';

        // Request parameters.
        var data = {
            id: rowData.id,
            do: 'OrdersOverviewAjax/GetInvoices',
            pageToken: jse.core.config.get('pageToken')
        };

        // Directly open a new tab (popup blocker workaround)
        var newTab = window.open('about:blank');

        $.ajax({ dataType: 'json', url: url, data: data }).done(function (response) {
            if (response.length) {
                // Get the file name from object with the highest ID within response array.
                var _response$ = response[0],
                    invoiceNumber = _response$.invoiceNumber,
                    orderId = _response$.orderId;

                // Invoice link parameters.

                var parameters = {
                    module: 'OrderAdmin',
                    action: 'showPdf',
                    type: 'invoice',
                    invoice_number: invoiceNumber,
                    order_id: orderId
                };

                // Open invoice
                newTab.location = jse.core.config.get('appUrl') + '/admin/request_port.php?' + $.param(parameters);
            } else {
                // No invoice found
                newTab.close();
                jse.libs.modal.showMessage(title, message);
            }
        });
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Bind table row actions.
        $this.on('click', 'tbody tr', _onTableRowClick).on('change', '.bulk-selection', _onBulkSelectionChange).on('change', 'input:checkbox', _onTableRowCheckboxChange).on('click', '.show-invoice', _onShowInvoiceClick).on('click', '.show-packing-slip', _onShowPackingSlipClick).on('click', '.create-invoice', _onCreateInvoiceClick).on('click', '.email-invoice', _onEmailInvoiceClick).on('click', '.email-order', _onEmailOrderClick).on('click', '.order-status.label', _onChangeOrderStatusClick).on('click', '.add-tracking-number', _onAddTrackingNumberClick);

        // Bind table row and bulk actions.
        $this.parents('.orders').on('click', '.btn-group .change-status', _onChangeOrderStatusClick).on('click', '.btn-group .cancel', _onCancelOrderClick).on('click', '.btn-group .bulk-email-order', _onBulkEmailOrderClick).on('click', '.btn-group .bulk-email-invoice', _onBulkEmailInvoiceClick).on('click', '.btn-group .bulk-download-invoice', _onBulkDownloadInvoiceClick).on('click', '.btn-group .bulk-download-packing-slip', _onBulkDownloadPackingSlipClick);

        // Bind custom events.
        $(document).on('create_missing_documents:proceed', '.modal.create-missing-documents', _onCreateMissingDocumentsProceed);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
